﻿/*
	VERSION:		2.0
	2.0				Full re-write of the original queueLoad.as,  with the intent of accomodating multiple files being loaded at the same time
	
	USAGE: 
		#include "functions/queueLoad.as"
		getFileList = _this.queueLoad("files\\fileList.txt", myCache, 5);
		getFileList.then = function( fileContents_str ){}
		
		
		#include "functions/queueLoad.as"
		_this.queueLoad( "charset\\image.png", myCache)
		.then = function( fileContents_bmp ){
			// 
		}// after loading()
		
		
		#include "functions/queueLoad.as"
		var myPath = "charset\\image.png";
		_this.queueLoad( myPath, myCache)
		.then = function( fileContents_bmp ){
			sendEvent("loadFile", {
				path: myPath, 
				image: fileContents_bmp
			});// sendEvent()
		}// after loading()
		
		
	WHAT THIS DOES: 
		Returns a promise for a requested file and queues the file to be loaded later.  (image, swf, or text)
		Be default, files are loaded 10 at a time.  But this setting can be changed by calling queueLoad() with its third parameter set to a different number.
		The same file won't get loaded more than once.  Repeated requests for a queued file returns the same promise for a single load of that file.
		Cached files are returned as immediately resolved promises.  (as if they loaded instantly)  (if a cache is provided)
		The most recently added files are loaded first.
		
		Automatically updates the provided cache when loading completes.  (if a cache is provided)
		
		This will automatically retry loading files that Flash forgets about  (When Flash does not return a success nor fail callback after 1/2 second of a load being attempted)
		After multiple unsuccessful retries, a file will resove as a failed load.  This prevents any infinitely pending promises.
		Retries have the lowest priority and are added to the end of the queue.  (inserted at the top of the array,  since high-priority items are pop() from the bottom)
		Of course, any of these items can still be prioritized again later if something specifically requests one of them via a queueLoad() call.
		Retries do not increase the overall load.  By re-adding items to the queue, the "loadAtOnce" setting it still enforced,  and re-loads are handled just like regular loads.
		Any external code waiting for a file is non-the-wiser.
		
	
	LEAKS: 
		runFunc()
		_this
		_this.queueLoad()
*/
if(!_this)		_this = this;

function runFunc( func ){
	return func.apply( _this, arguments.slice() );
}// runFunc()



runFunc(function(){
	// VOW is automatically local-only
	#include "functions/VOW.as"
	var loadFile;			// local-only
	#include "functions/VOW/loadFile.as"
	var loadBitmap;		// local-only
	#include "functions/VOW/loadBitmap.as"
	
	var loadAtOnce  = 10;		// 15 yields great times,  but might be pushing reliability  (Retries are never desirable)
	var itemsBeingLoaded = 0;
	var cache = undefined;		// no cache by default
	var queue = [];
	var retryDelay_ms = 500;
	var retries = 3;
	
	_this.queueLoad = function( path, new_cache, new_loadAtOnce ){
		// handle params
		if( !path )														return VOW.make().keep( undefined );
		if( new_cache !== undefined )					cache = new_cache;
		if( new_loadAtOnce > 0 )		loadAtOnce = new_loadAtOnce;
		
		// already loaded?
		if( cache[ path ] ){
			var output = VOW.make().keep( cache[ path ] );
			return output;
		}
		
		// already loading?
		var foundIndex = getQueueIndex[ path ];
		if( foundIndex !== undefined ){
			prioritizeIndex( foundIndex );
			return queue[ foundIndex ].prom;
		}
		
		// else, not in queue...
		// add this item to the load-queue
		var newProm = enqueuePath( path );
		// load this item if we're not already at maximum loading-capacity
		if( itemsBeingLoaded < loadAtOnce )		loadNextItem();
		
		return newProm;
	}// queueLoad()
	
	
	
	function getQueueIndex( findPath ){
		for(var q=queue.length-1; q>=0; q--){
			// if this entry already exists in the queue,  return this item's promise
			if( queue[ q ].path === findPath )		return  queue[ q ].prom;
		}// for each:  queue item
		
		// if:  not found
		return undefined;
	}// getQueueIndex()
	
	
	
	// make this item the next one to load
	function prioritizeIndex( thisIndex ){
		// catch invalid index's
		if( queue[ thisIndex ] === undefined )		return;
		
		// remove from middle of queue
		var removedItem = queue.splice( thisIndex, 1 )[ 0 ];
		// add to end of queue
		queue.push( removedItem );
	}// priotitizeIndex()
	
	
	
	// add this item to the end of the queue as the next item to be loaded
	function enqueuePath( newPath ){
		var newItem = {
			path: newPath, 
			prom: VOW.make(), 
			retry: 0
		}
		// add to the list of things to load
		queue.push( newItem );
		// return this item's load-promise
		return newItem.prom;
	}// enqueuePath()
	
	
	
	// add this item to the beginning of the queue as the last item to be loaded
	function reQueueItem( oldItem ){
		queue.splice( 0, 0, oldItem );
	}// reQueueItem()
	
	
	
	// load the item at the end of the queue  (typically the last item added)
	function loadNextItem(){
		// abort if nothing to load
		if( queue.length === 0 )		return;
		
		// remove the last item from the queue
		var nextItem = queue.pop();
		// we're loading one more file than before
		itemsBeingLoaded++;
		
		// already loaded?
		var cachedValue = cache[ nextItem.path ];
		if( cachedValue ){
			// we're no longer loading this file
			itemsBeingLoaded--;
			// announce the cached result
			if( nextItem.prom.getStatus() === "pending" )		nextItem.prom.keep( cachedValue );
			// load the next files
			loadMoreFiles();
			// there's nothing left to do with this file
			return;
		}
		
		// skip this file if path is invalid
		if( !nextItem.path )		loadFailure();
		
		var fileType = getFileType( nextItem.path );
		
		
		if( fileType === "image" ){
			// 1st attempt
			loadBitmap( nextItem.path )
			.then( function( contents_bmp ){
				// we're no longer loading this file
				itemsBeingLoaded--;
				
				// if:  invalid content  >>  fail
				var isBitmap = (contents_bmp instanceof flash.display.BitmapData);
				if( !isBitmap ){
					if( nextItem.prom.getStatus() === "pending" )		nextItem.prom.doBreak( undefined );
					return;
				}// if:  invalid content  >>  fail
				// else... load was successful
				
				// update the cache
				updateCache( contents_bmp );
				
				// load succeeded
				if( nextItem.prom.getStatus() === "pending" )		nextItem.prom.keep( contents_bmp );
				
				loadMoreFiles();
			}, loadFailure );// done loading
			
			// 2nd attempt
			setTimeout(function(){
				retry();
			}, retryDelay_ms);
		}// if:  image
		
		
		if( fileType === "text" ){
			// 1st attempt
			loadFile( nextItem.path )
			.then( function( contents_str ){
				// we're no longer loading this file
				itemsBeingLoaded--;
				
				// if:  invalid content  >>  fail
				var isString = ( typeof(contents_str) === "string" );
				if( !isString ){
					if( nextItem.prom.getStatus() === "pending" )		nextItem.prom.doBreak( undefined );
					return;
				}// if:  invalid content  >>  fail
				// else... load was successful
				
				// update the cache
				updateCache( contents_str );
				
				// load succeeded
				if( nextItem.prom.getStatus() === "pending" )		nextItem.prom.keep( contents_str );
				
				loadMoreFiles();
			}, loadFailure );// done loading
			
			// 2nd attempt
			setTimeout(function(){
				retry();
			}, retryDelay_ms);
		}// if:  text
		
		
		function retry(){
			// This file has already loaded / failed to load
			if( nextItem.prom.getStatus() !== "pending" )		return;
			
			// else... This file-load has been forgotten by Flash
			// if:  this file has already been attempted multiple times
			if( nextItem.retry >= retries ){
				// then:  give up
				loadFailure();
				return;
			}// if:  this file has already been attempted multiple times
			
			// else... This file-load has been forgotten by Flash
			// retry
			nextItem.retry++;
			// re-add this item to the queue,  at the end
			reQueueItem( nextItem );
			// load this item now if we're not already at maximum loading-capacity
			loadMoreFiles();
		}// retry()
		
		
		function loadFailure(){
			// load failure
			// we're no longer loading this file
			itemsBeingLoaded--;
			// announce that this item failed
			if( nextItem.prom.getStatus() === "pending" )		nextItem.prom.doBreak( undefined );
			// try the next files
			loadMoreFiles();
		}// loadFailure()
		
		
		// store the result in the cache
		function updateCache( contents ){
			// abort if already cached
			if( cache[ nextItem.path ] !== undefined )		return;
			// abort if there is no cache
			if( !cache )		return;
			// store the loaded result in the cache
			cache[ nextItem.path ] = contents;
		}// updateCache()
		
		
		// load as many files as you can
		function loadMoreFiles(){
			var loadThisMany = ( loadAtOnce - itemsBeingLoaded );
			// avoid loading infinite files
			if( loadThisMany < 0 )		return;
			// load as many files as we're allowed to
			for(var l=0; l<loadThisMany; l++){
				loadNextItem();
			}// for:  loadThisMany
			
		}// loadMoreFiles()
		
	}// loadNextItem()
	
	
	
	function getFileType( path ){
		// abort if invalid path
		var pathIsString = ( typeof(path) === "string" );
		if( !pathIsString )		return undefined;
		
		// abort if empty path
		if( !path )		return undefined;
		
		// get file extension
		var extAt = path.lastIndexOf( '.' ) + 1;
		var ext = path.substr( extAt );
		// abort if no file extension
		if( !ext )		return undefined;
		
		// interpret file extension
		switch( ext ){
			case "jpg": 
			case "jpeg": 
			case "png": 
			case "gif": 
				return "image";
				break;
			default: 
				return "text";
		}
	}// getFileType()
	
	
});// runFunc()